%% Dear readers, if this platform is useful for you, please kindly cite:
% S. Wang, Y.-C. Wu, and M. Xia, Achieving global optimality in wirelessly-powered multi-antenna TWRC with lattice codes, in Proc. IEEE ICASSP16, Shang Hai, China, Mar. 2016, pp. 3556-3560.

% You may also be interested in the following references.
% [1] S. Boyd and L. Vandenberghe, Convex Optimization. Cambridge, U.K.: Cambridge Univ. Press, 2004.
% [2] Z. Wen, S. Wang, C. Fan, and W. Xiang, Joint transceiver and power splitter design over two-way relaying channel with lattice codes and energy harvesting, IEEE Commun. Lett., vol. 18, no. 11, pp. 2039-2042, Nov. 2014.

addpath('.\CVX');
cvx_setup; % setup CVX, a MATLAB software for solving convex optimization problems

clear all;
% Information for the platform
disp('***** This platform is used to simulate the algorithm in the following paper.');
disp('Authors: S. Wang, Y.-C. Wu, and M. Xia,');
disp('Title: Achieving global optimality in wirelessly-powered multi-antenna TWRC with lattice codes');
disp('Publication: Proc. IEEE ICASSP16, Shang Hai, China, Mar. 2016, pp. 3556-3560.');

%% Initialization
load('H_K1.mat'); % load all the channels
sigma2=-35; % noise power in dBm
Pn=0.001*10.^(sigma2/10); % noise power in W
Pc = 5; % circuit power in dBm
Pc=0.001*10.^(Pc/10); % circuit power in W
R_vec = 1:0.2:3; % data-rate QoS in bit/Hz, for Fig. 2b in ICASSP'16
P_1=zeros(monte, length(R_vec)); % total transmit power of scheme 1: optimal solution in ICASSP'16
P_2=zeros(monte, length(R_vec)); % total transmit power of scheme 2: block coordinate descent method
P_3=zeros(monte, length(R_vec)); % total transmit power of scheme 3: fixed power splitting ratio 0.5

%% Main part
tic; % time counter
for s=1:1:length(R_vec)
    R_12=R_vec(s); % data-rate QoS from user 1 to user 2
    R_21=R_vec(s); % data-rate QoS from user 2 to user 1
    QoS_12 = 2.^(2.*R_12); % due to logarithm in Shannon capacity
    QoS_21 = 2.^(2.*R_21);
    QoS_ir=[QoS_12 QoS_21]'; % uplink QoS target from user i to relay
    QoS_ri=[QoS_21 QoS_12]'; % downlink QoS target from relay to user i
   
    fprintf('Start simulating data-rate QoS equal to %d.\n', R_vec(s));
    % Monte-carlo simulation
    for m = 1:monte
        if mod(m,50)==0
            disp('50 monte carlo finished!');
            toc; % show time every 50 runs
        end
        
        % load channels for the m^th run
        h=h_m(:,:,:,:,m);
        g=g_m(:,:,:,:,m);
                
        % optimal solution in ICASSP'16
        P_1(m,s)= scheme1(h, g, Pc, Pn, QoS_ir, QoS_ri);
        % block coordinate descent method
        P_2(m,s) = scheme2(N, h, g, Pc, Pn, QoS_ir, QoS_ri);
        % fixed power splitting ratio 0.5
        P_3(m,s)= scheme3(h, g, Pc, Pn, QoS_ir, QoS_ri);
                
    end 
end

%% Calculate results
cc=sum(P_1(:,1)~=0); % count all the non-zero powers
Avg_P1 = sum(P_1,1)./cc; % average transmit power in W for scheme 1
Avg_P2 = sum(P_2,1)./cc; % scheme 2
Avg_P3 = sum(P_3,1)./cc; % scheme 3
Avg_P1_dBm = 10*log10(Avg_P1./0.001); % average transmit power in dBm for scheme 1
Avg_P2_dBm = 10*log10(Avg_P2./0.001); % scheme 2
Avg_P3_dBm = 10*log10(Avg_P3./0.001); % scheme 3








